import sqlite3
import os
import traceback

class AIDatabase:
    """AI插件独立数据库类"""
    
    def __init__(self):
        self.conn = None
        self.db_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'ai_plugin.db')
        self.init_db()
    
    def create_tables(self):
        """创建AI相关的数据库表"""
        # 1. AI设置表
        self.conn.execute("""
        CREATE TABLE IF NOT EXISTS ai_settings(
            key TEXT PRIMARY KEY,
            value TEXT
        )""")
        
        # 2. AI历史记录表
        self.conn.execute("""
        CREATE TABLE IF NOT EXISTS ai_history(
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            ts TEXT,
            role TEXT,
            content TEXT,
            tags TEXT
        )
        """)
        self.conn.execute("CREATE INDEX IF NOT EXISTS idx_ai_tags ON ai_history(tags)")
        self.conn.execute("CREATE INDEX IF NOT EXISTS idx_ai_ts ON ai_history(ts)")
        
        self.conn.commit()
    
    def init_db(self):
        """初始化数据库连接"""
        try:
            # 安全地关闭已存在的连接
            if hasattr(self, 'conn') and self.conn is not None:
                try:
                    self.conn.close()
                except Exception as e:
                    print(f"关闭旧数据库连接时出错: {e}")
                self.conn = None
                
            # 尝试打开数据库
            self.conn = sqlite3.connect(self.db_path)
            self.conn.execute("PRAGMA foreign_keys = ON")
            
            # 创建/更新表结构
            self.create_tables()
            
            print(f"AI数据库连接初始化成功: {self.db_path}")
        except sqlite3.OperationalError as e:
            print(f"无法打开AI数据库文件：{e}")
            self.handle_db_error()
    
    def handle_db_error(self):
        """处理数据库错误"""
        # 尝试的备选路径列表
        alternative_paths = [
            os.path.join(os.path.expanduser('~'), 'ai_plugin.db'),  # 用户目录
            os.path.join(os.getenv('TEMP', ''), 'ai_plugin.db')  # 临时目录（最后选项）
        ]
        
        # 尝试所有备选路径
        for path in alternative_paths:
            try:
                # 确保目录存在
                os.makedirs(os.path.dirname(os.path.abspath(path)), exist_ok=True)
                # 尝试连接
                self.conn = sqlite3.connect(path)
                self.conn.execute("PRAGMA foreign_keys = ON")
                
                # 创建表
                self.create_tables()
                
                print(f"成功使用备选路径：{path}")
                self.db_path = path
                return  # 成功后返回
            except Exception as e:
                print(f"尝试路径 {path} 失败: {e}")
                continue
                
        # 所有路径都失败
        print("所有AI数据库路径尝试都失败")
        traceback.print_exc()
    
    def set_setting(self, key, value):
        """设置AI配置"""
        self.conn.execute("INSERT OR REPLACE INTO ai_settings(key,value) VALUES(?,?)",
                          (key, str(value)))
        self.conn.commit()
    
    def get_setting(self, key, default=None):
        """获取AI配置"""
        try:
            c = self.conn.execute("SELECT value FROM ai_settings WHERE key=?", (key,))
            row = c.fetchone()
            return row[0] if row else default
        except Exception:
            return default
    
    def add_ai_message(self, role, content, tags=None):
        """添加AI消息记录"""
        try:
            from PyQt5.QtCore import QDateTime
            ts = QDateTime.currentDateTime().toString("yyyy-MM-dd HH:mm:ss")
            cur = self.conn.execute("INSERT INTO ai_history(ts,role,content,tags) VALUES(?,?,?,?)", 
                                  (ts, role, content, tags or ""))
            self.conn.commit()
            return cur.lastrowid
        except Exception:
            try:
                self.conn.commit()
            except Exception:
                pass
            return None
    
    def list_ai_history(self, limit=100):
        """获取AI历史记录"""
        try:
            cur = self.conn.execute("SELECT id,ts,role,content,tags FROM ai_history ORDER BY id DESC LIMIT ?", 
                                  (int(limit),))
            return cur.fetchall()
        except Exception:
            return []
    
    def get_ai_session_messages(self, session_id, limit=1000):
        """获取AI会话消息"""
        try:
            cur = self.conn.execute(
                "SELECT id,ts,role,content,tags FROM ai_history WHERE tags LIKE ? ORDER BY id ASC LIMIT ?",
                (f"%session:{session_id}%", int(limit))
            )
            return cur.fetchall()
        except Exception:
            return []
    
    def delete_ai_session(self, session_id):
        """删除AI会话"""
        try:
            before = self.conn.total_changes
            self.conn.execute("DELETE FROM ai_history WHERE tags LIKE ?", (f"%session:{session_id}%",))
            after = self.conn.total_changes
            if after == before:
                try:
                    sid = int(str(session_id))
                    self.conn.execute("DELETE FROM ai_history WHERE id=?", (sid,))
                except Exception:
                    pass
            self.conn.commit()
        except Exception:
            self.conn.commit()
    
    def delete_ai_message(self, msg_id):
        """删除AI消息"""
        try:
            self.conn.execute("DELETE FROM ai_history WHERE id=?", (msg_id,))
            self.conn.commit()
            return True
        except Exception:
            return False
    
    def clear_all_data(self):
        """清空所有AI数据"""
        try:
            # 清空所有表
            self.conn.execute("DELETE FROM ai_settings")
            self.conn.execute("DELETE FROM ai_history")
            self.conn.commit()
            return True
        except Exception as e:
            print(f"清空AI数据失败: {e}")
            return False
    
    def close(self):
        """安全关闭数据库连接"""
        try:
            if hasattr(self, 'conn') and self.conn is not None:
                # 提交任何未提交的事务
                self.conn.commit()
                # 关闭连接
                self.conn.close()
                self.conn = None  # 避免重复关闭
                print("AI数据库连接已关闭")
        except Exception as e:
            print(f"关闭AI数据库连接时出错: {e}")
            traceback.print_exc()